/* Part of the culibs project, <http://www.eideticdew.org/culibs/>.
 * Copyright (C) 2009  Petter Urkedal <urkedal@nbi.dk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef CU_INSTALLDIRS_H
#define CU_INSTALLDIRS_H

#include <cu/fwd.h>

CU_BEGIN_DECLARATIONS
/** \defgroup cu_installdirs_h cu/installdirs.h: Installation Directories
 ** @{ \ingroup cu_util_mod
 **
 ** This provides a way for an application or library to manage its
 ** installation directories.  Prefix substitutions are handled by allowing
 ** each directory to refer to any of the others as its prefix.
 **
 ** A C source file which defines \ref cu_installdirs_t with preliminary
 ** initialisation for a package is generated by invoking the Autoconf macro
 ** \code
 ** CUAC_CONFIG_INSTALLDIRS(foo_installdirs.c, foo_installdirs)
 ** \endcode
 ** Then \c foo_installdirs can be used as
 ** \code
 ** #include <cu/installdirs.h>
 **
 ** extern cu_installdirs_t foo_installdirs;
 **
 ** void foo_init(void)
 ** {
 **     // ...
 **     cu_installdirs_finish(&foo_installdirs);
 **     // ...
 ** }
 **
 ** char const *foo_get_installdir(cu_installdir_key_t key)
 ** {
 **     return foo_installdirs[key].dir;
 ** }
 ** \endcode
 **/

/** A reference to one of the installation directories in \ref
 ** cu_installdirs_t.  In the following, typecal values for a prefix of \c /usr
 ** are shown in paretheses. */
typedef enum {
    /* Note!  These must match cuac_config_installdirs.m4. */
    CU_INSTALLDIR_PREFIX,	/**< The \c ${prefix} (\c /usr) */
    CU_INSTALLDIR_EXEC_PREFIX,	/**< The \c ${exec_prefix} (\c /usr) */

    CU_INSTALLDIR_BINDIR,	/**< The \c ${bindir} (\c /usr/bin) */
    CU_INSTALLDIR_SBINDIR,	/**< The \c ${sbindir} (\c /usr/sbin) */
    CU_INSTALLDIR_LIBEXECDIR,	/**< The \c ${libexecdir} (\c /usr/libexec) */
    CU_INSTALLDIR_SYSCONFDIR,	/**< The \c ${sysconfdir} (\c /etc) */
    CU_INSTALLDIR_SHAREDSTATEDIR,/**< The \c ${sharedstatedir} (\c /var/com) */
    CU_INSTALLDIR_LOCALSTATEDIR,/**< The \c ${localstatedir} (\c /var/lib) */
    CU_INSTALLDIR_LIBDIR,	/**< The \c ${libdir} (\c /usr/lib) */
    CU_INSTALLDIR_INCLUDEDIR,	/**< The \c ${includedir} (\c /usr/include) */
    CU_INSTALLDIR_DATAROOTDIR,	/**< The \c ${datarootdir} (\c /usr/share) */
    CU_INSTALLDIR_DATADIR,	/**< The \c ${datadir} (\c /usr/share) */
    CU_INSTALLDIR_INFODIR,	/**< The \c ${infodir} (\c /usr/share/info) */
    CU_INSTALLDIR_LOCALEDIR,/**< The \c ${localedir} (\c /usr/share/locale) */
    CU_INSTALLDIR_MANDIR,   /**< The \c ${mandir} (\c /usr/share/man) */
    CU_INSTALLDIR_DOCDIR,   /**< The \c ${docdir} (\c /usr/share/doc/foo-1.0)*/
    CU_INSTALLDIR_HTMLDIR,
    CU_INSTALLDIR_DVIDIR,
    CU_INSTALLDIR_PDFDIR,
    CU_INSTALLDIR_PSDIR,

    CU_INSTALLDIR_NONE
} cu_installdir_key_t;

typedef struct cu_installdir *cu_installdir_t;
struct cu_installdir
{
    char const *name;
    cu_installdir_key_t key;

    cu_installdir_key_t prefix_key;
    char const *suffix;
    char const *dir;
};

/** An array describing the installation directories of a package. */
typedef struct cu_installdir cu_installdirs_t[CU_INSTALLDIR_NONE + 1];

/** Set the installation directory indicated by \a key to \a dir.  The
 ** directory may be a prefix for other directories, in which case \ref
 ** cu_installdirs_finish will substitute it.  This function must be called
 ** before \ref cu_installdirs_finish. */
void cu_installdirs_set(cu_installdirs_t installdirs, cu_installdir_key_t key,
			char const *dir);

/** If found, sets the configuration directory named \a name in lowercase to \a
 ** dir and returns true, else returns false.  This function must be called
 ** before \ref cu_installdirs_finish. */
cu_bool_t cu_installdirs_set_byname(cu_installdirs_t installdirs, char const *name,
				    char const *dir);

/** Set \a installdirs according to environment variables named \a var_prefix
 ** followed by the common installation directory names.  If \a var_prefix
 ** starts with an uppercase, the directory names are uppercased as well.  E.g.
 ** calling <tt>cu_installdirs_set_byenv(&mydirs, "MYAPP_")</tt> will pick up
 ** \c $MYAPP_PREFIX, \c $MYAPP_EXEC_PREFIX, \c $MYAPP_SYSCONFDIR, etc.  This
 ** function must be called before \ref cu_installdirs_finish. */
void cu_installdirs_set_byenv(cu_installdirs_t installdirs, char const *var_prefix);

/** Completes the \c dir fields of the application's installation directory
 ** mapping, assuming the first two components have been initialised.  This
 ** function is typically called during program initialisation. */
void cu_installdirs_finish(cu_installdirs_t installdirs);

/** Prints out \a installdirs. */
void cu_installdirs_dump(cu_installdirs_t installdirs);

/** The culibs installation directory \a key. */
char const *cuconf_get_installdir(cu_installdir_key_t key);

/** @} */
CU_END_DECLARATIONS

#endif
