/*-*-C-*-
 * Copyright 2004--2005  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CU_INT_H
#define CU_INT_H

#include <cu/fwd.h>
#include <cu/conf.h>
#include <cu/debug.h>
#include <stdint.h>

CU_BEGIN_DECLARATIONS
/*!\defgroup cu_int cu/int.h: Integer Functions
 * @{ \ingroup cu_mod */

#if CUCONF_SIZEOF_INT == 4
#  define CUP_UINT_NAME(prefix, name) prefix##uint32_##name
#elif CUCONF_SIZEOF_INT == 8
#  define CUP_UINT_NAME(prefix, name) prefix##uint64_##name
#else
#  error Unexpected CUCONF_SIZEOF_INT
#endif

#if CUCONF_SIZEOF_LONG == 4
#  define CUP_ULONG_NAME(prefix, name) prefix##uint32_##name
#elif CUCONF_SIZEOF_LONG == 8
#  define CUP_ULONG_NAME(prefix, name) prefix##uint64_##name
#else
#  error Unexpected CUCONF_SIZEOF_LONG
#endif


/*!Returns a bitmask the uppermost non-zero bit in \a x and downwards. */
uint_fast8_t cu_uint8_dcover(uint_fast8_t x);

/*!\copydoc cu_uint8_dcover */
uint_fast16_t cu_uint16_dcover(uint_fast16_t x);

/*!\copydoc cu_uint8_dcover */
uint_fast32_t cu_uint32_dcover(uint_fast32_t x);

/*!\copydoc cu_uint8_dcover */
uint_fast64_t cu_uint64_dcover(uint_fast64_t x);

/*!\copydoc cu_uint8_dcover */
CU_SINLINE unsigned int cu_uint_dcover(unsigned int x)
{ return CUP_UINT_NAME(cu_,dcover)(x); }

/*!\copydoc cu_uint8_dcover */
CU_SINLINE unsigned long cu_ulong_dcover(unsigned long x)
{ return CUP_ULONG_NAME(cu_,dcover)(x); }

/*!Returns a bitmask from the lowermost non-zero bit in \a x and upwards. */
CU_SINLINE unsigned int
cu_uint_ucover(unsigned int x) { return x | ~(x - 1); }

/*!\copydoc cu_uint_ucover */
CU_SINLINE unsigned long
cu_ulong_ucover(unsigned long x) { return x | ~(x - 1L); }


/*!A bitmask which covers the lowermost continuous zeros of \a x. */
CU_SINLINE unsigned int
cu_uint_lzeros(unsigned int x) { return ~x & (x - 1); }

/*!\copydoc cu_uint_lzeros */
CU_SINLINE unsigned long
cu_ulong_lzeros(unsigned long x) { return ~x & (x - 1L); }

/*!A bitmask which covers the uppermost continuous zeroes of \a x. */
CU_SINLINE unsigned int
cu_uint_uzeros(unsigned int x) { return ~cu_uint_dcover(x); }

/*!\copydoc cu_uint_uzeros */
CU_SINLINE unsigned long
cu_ulong_uzeros(unsigned long x) { return ~cu_ulong_dcover(x); }


/*!Returns 2<sup>⌈log<sub>2</sub> <i>x</i>⌉</sup>. */
CU_SINLINE uint_fast8_t cu_uint8_exp2_ceil_log2(uint_fast8_t x)
{ return cu_uint8_dcover(x - 1) + 1; }

/*!\copydoc cu_uint8_exp2_ceil_log2 */
CU_SINLINE uint_fast16_t cu_uint16_exp2_ceil_log2(uint_fast16_t x)
{ return cu_uint16_dcover(x - 1) + 1; }

/*!\copydoc cu_uint8_exp2_ceil_log2 */
CU_SINLINE uint_fast32_t cu_uint32_exp2_ceil_log2(uint_fast32_t x)
{ return cu_uint32_dcover(x - UINT32_C(1)) + UINT32_C(1); }

/*!\copydoc cu_uint8_exp2_ceil_log2 */
CU_SINLINE uint_fast64_t cu_uint64_exp2_ceil_log2(uint_fast64_t x)
{ return cu_uint64_dcover(x - UINT64_C(1)) + UINT64_C(1); }

/*!\copydoc cu_uint8_exp2_ceil_log2 */
CU_SINLINE unsigned int
cu_uint_exp2_ceil_log2(unsigned int x)
{ return cu_uint_dcover(x - 1) + 1; }

/*!\copydoc cu_uint8_exp2_ceil_log2 */
CU_SINLINE unsigned long
cu_ulong_exp2_ceil_log2(unsigned long x)
{ return cu_ulong_dcover(x - 1L) + 1L; }


/*!Returns the number of bits in \a x which are set. */
unsigned int cu_uint8_bit_count(uint_fast8_t x);
/*!Returns the number of bits in \a x which are set. */
unsigned int cu_uint16_bit_count(uint_fast16_t x);
/*!Returns the number of bits in \a x which are set. */
unsigned int cu_uint32_bit_count(uint_fast32_t x);
/*!Returns the number of bits in \a x which are set. */
unsigned int cu_uint64_bit_count(uint_fast64_t x);

/*!Returns the number of bits in \a x which is set. */
CU_SINLINE unsigned int cu_uint_bit_count(unsigned int x)
{ return CUP_UINT_NAME(cu_,bit_count)(x); }

/*!Returns the number of bits in \a x which is set. */
CU_SINLINE unsigned long cu_ulong_bit_count(unsigned long x)
{ return CUP_ULONG_NAME(cu_,bit_count)(x); }


/*!Returns the exponent of the uppermost non-zero bit in \a x, that is
 * ⌊log<sub>2</sub> <i>x</i>⌋. */
unsigned int cu_uint8_floor_log2(uint_fast8_t x);
/*!\copydoc cu_uint8_floor_log2*/
unsigned int cu_uint16_floor_log2(uint_fast16_t x);
/*!\copydoc cu_uint8_floor_log2*/
unsigned int cu_uint32_floor_log2(uint_fast32_t x);
/*!\copydoc cu_uint8_floor_log2*/
unsigned int cu_uint64_floor_log2(uint_fast64_t x);
/*!\copydoc cu_uint8_floor_log2*/
unsigned int cu_uint_floor_log2(unsigned int x);
/*!\copydoc cu_uint8_floor_log2*/
unsigned int cu_ulong_floor_log2(unsigned long x);

/*!Returns the exponent of the lowermost non-zero bit in \a x. */
CU_SINLINE unsigned int
cu_uint8_log2_lowbit(uint_fast8_t x)
{
    cu_debug_assert(x);
    return cu_uint8_bit_count(~x & (x - 1));
}
/*!\copydoc cu_uint8_log2_lowbit*/
CU_SINLINE unsigned int
cu_uint16_log2_lowbit(uint_fast16_t x)
{
    cu_debug_assert(x);
    return cu_uint16_bit_count(~x & (x - 1));
}
/*!\copydoc cu_uint8_log2_lowbit*/
CU_SINLINE unsigned int
cu_uint32_log2_lowbit(uint_fast32_t x)
{
    cu_debug_assert(x);
    return cu_uint32_bit_count(~x & (x - 1));
}
/*!\copydoc cu_uint8_log2_lowbit*/
CU_SINLINE unsigned int
cu_uint64_log2_lowbit(uint_fast64_t x)
{
    cu_debug_assert(x);
    return cu_uint64_bit_count(~x & (x - 1));
}

/*!\copydoc cu_uint8_log2_lowbit*/
CU_SINLINE unsigned int
cu_uint_log2_lowbit(unsigned int x)
{ return CUP_UINT_NAME(cu_,log2_lowbit)(x); }

/*!\copydoc cu_uint8_log2_lowbit*/
CU_SINLINE unsigned int
cu_ulong_log2_lowbit(unsigned long x)
{ return CUP_ULONG_NAME(cu_,log2_lowbit)(x); }


/* Network Byte Order and Host Byte Order Conversion
 * ------------------------------------------------- */

#ifdef CUCONF_WORDS_BIGENDIAN

/*!Returns \a i in network byte order */
CU_SINLINE uint16_t cu_uint16_hton(uint16_t i) { return i; }
/*!\copydoc cu_uint16_hton */
CU_SINLINE uint32_t cu_uint32_hton(uint32_t i) { return i; }

#else

CU_SINLINE uint16_t cu_uint16_hton(uint16_t i)
{
    return (i >> 8) | ((i & 0xff) << 8);
}
CU_SINLINE uint32_t cu_uint32_hton(uint32_t i)
{
    return (i >> 24) | ((i & 0xff) << 24)
	| ((i & 0xff0000) >> 8) | ((i & 0xff00) << 8);
}

#endif
#define cu_uint16_ntoh cu_uint16_hton
#define cu_uint32_ntoh cu_uint32_hton

/*!@}*/
CU_END_DECLARATIONS

#endif
