/*-*-C-*-
 * Copyright 2000--2004  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#define CU_DEBUG 1
#include <stdio.h>
#include <cu/conf.h>
#include <cu/memory.h>
#include <cu/memory_bits.h>
#include <cu/debug.h>
#include <cu/tstate.h>
#include <cu/hcons.h>
#include <gc/gc_mark.h>
#include <assert.h>
#if 0
#  include <gc/private/gc_priv.h>
#else
#  ifndef PTRFREE
#    define PTRFREE 0
#  endif
#  ifndef NORMAL
#    define NORMAL 1
#  endif
#  ifndef ATOMIC
#    define UNCOLLECTABLE 2
#  endif
#endif

static cu_bool_t memory_init_done = 0;
#ifdef CUCONF_DEBUG_MEMORY
static cu_bool_t enable_dbginfo = 0;
#endif
#if 0
#ifdef CUCONF_HAVE_RECLAIM_NOTIFICATION
static int object_kind;
static void **object_freelist;
#endif
#endif

static void
out_of_memory(size_t size)
{
    fprintf(stderr, "Allocation of %ld bytes failed.\n", (long)size);
    abort();
}

static void (*cuP_raise_out_of_memory)(size_t size) = out_of_memory;

void
cu_raise_out_of_memory(size_t size)
{
    (*cuP_raise_out_of_memory)(size);
}

void
cu_regh_out_of_memory(void (*f)(size_t))
{
    cuP_raise_out_of_memory = f;
}

cuP_dbginfo_t db_header_first = NULL;

void *
cuP_apply_n(void (*f)(), void *ptr, size_t step, size_t n)
{
    void *p = ptr;
    while (n--) {
	f(ptr);
	p = (char*)p + step;
    }
    return ptr;
}

void
cuP_debug_show_all_segments()
{
    cuP_dbginfo_t p = db_header_first;
    if (!p)
	fprintf(stderr, "[debug] All segment allocated with "
		"cu_malloc are freed.\n");
    else {
	fprintf(stderr, "[debug] The following segments are unfreed.\n");
	while (p) {
	    fprintf(stderr, "%s:%d: Unfreed memory allocated here.\n",
		    p->name, p->line);
	    p = p->next;
	}
	fprintf(stderr, "\n");
    }
}

void
cuP_debug_show_segment(void *p0)
{
    cuP_dbginfo_t p = p0;
    --p;
    fprintf(stderr, "Memory segment allocated at %s:%d.\n",
	    p->name, p->line);
}


/* Plain Dynamic Memory
 * ==================== */

void *
cuP_debug_malloc(size_t size, const char *name, int line)
{
    cuP_dbginfo_t p
	= malloc(size + CU_ALIGNED_SIZEOF(struct cuP_dbginfo_s));
    p->name = name;
    p->line = line;
    p->prev = NULL;
    p->next = db_header_first;
    p->cminfo_chain = NULL;
    if (db_header_first) db_header_first->prev = p;
    db_header_first = p;
    if (cu_debug_is_enabled(cu_debug_show_alloc_flag))
	fprintf(stderr, "%s:%d: cu_malloc() = %p.\n",
		p->name, p->line, CU_ALIGNED_PTR_END(p));
    return CU_ALIGNED_PTR_END(p);
}

void
cuP_debug_mfree(void *p1)
{
    cuP_dbginfo_t p = p1;
    --p;
    if (p->prev) p->prev->next = p->next;
    if (p->next) p->next->prev = p->prev;
    if (db_header_first == p)
	db_header_first = p->next;
    if (cu_debug_is_enabled(cu_debug_show_alloc_flag))
	fprintf(stderr, "%s:%d: cu_mfree(%p).\n", p->name, p->line, p1);
}


/* Garbage Collected Memory
 * ======================== */

static ptrdiff_t cuD_gc_base_shift = -1;

#ifdef CUCONF_DEBUG_MEMORY
void *
cu_gc_base(void *ptr)
{
    /* This compensates for the GC_debug_* allocators.  It isn't
     * needed at the moment, since we can't use these allocators with
     * the kind-general local alloc. */
    if (cuD_gc_base_shift == -1)
	cu_bugf("Library is not initialised.");
    ptr = GC_base(ptr);
    if (ptr == NULL)
	return ptr;
    else
	return ptr + cuD_gc_base_shift;
}

CU_SINLINE void
cuP_dbginfo_cct(cuP_dbginfo_t hdr, char const *file, int line)
{
    hdr->name = file;
    hdr->line = line;
    hdr->prev = NULL;
    hdr->next = NULL;
    hdr->cminfo_chain = NULL;
}
#endif

#define DBG_MAX_ALLOC_SIZE 100000000

CU_SINLINE void
cuD_check_size(size_t size, char const *file, int line)
{
    if (size > DBG_MAX_ALLOC_SIZE) {
	cu_errf("Trying to allocate %d bytes, which is above debug limit.",
		 size);
	cu_debug_unreachable();
    }
}

#ifdef CUCONF_DEBUG_MEMORY

void *cu_galloc_D(size_t size, char const *file, int line)
{
    void *p;
    if (enable_dbginfo) {
	size += sizeof(struct cuP_dbginfo_s);
	p = GC_debug_malloc(size, file, line);
	if (p == NULL)
	    cu_raise_out_of_memory(size);
	ccg_dbginfo_cct(p, file, line);
	return p + sizeof(struct cuP_dbginfo_s);
    }
    else {
	p = GC_debug_malloc(size, file, line);
	if (p == NULL)
	    cu_raise_out_of_memory(size);
    }
}

void
cu_gfree_D(void *ptr, char const *file, int line)
{
    void *base;
    if (!ptr)
	return;
    base = cu_gc_base(ptr);
    if (base != ptr)
	cu_bugf("Invalid pointer passed to cu_gfree: %p (base = %p)",
		 ptr, base);
    if (enable_dbginfo)
	ptr -= sizeof(struct cuP_dbginfo_s);
    GC_debug_free(ptr);
}

#ifndef CUCONF_HAVE_GC_MALLOC_ATOMIC_UNCOLLECTABLE
void
cu_gfree_au_D(void *ptr, char const *file, int line)
{
    void *base;
    if (!ptr)
	return;
    base = cu_gc_base(ptr);
    if (base != ptr)
	cu_bugf_fl(file, line,
		    "Invalid pointer passed to cu_gfree_au: %p (base = %p)",
		    ptr, base);
    if (enable_dbginfo)
	ptr -= sizeof(struct cuP_dbginfo_s);
    free(ptr);
}
#endif

#endif /* CUCONF_DEBUG_MEMORY */



/* Other GC facilities
 * ------------------- */

void *
cuP_weakptr_get_locked(void *link)
{
    if (*(cu_hidden_ptr_t *)link)
	return cu_reveal_ptr(*(cu_hidden_ptr_t *)link);
    else
	return NULL;
}

void
cu_debug_meminfo_at(void *ptr)
{
    cuP_dbginfo_t meminfo = cuP_dbginfo(ptr);
    if (meminfo)
	fprintf(stderr, "%s:%d: Memory at %p allocated here.\n",
		meminfo->name, meminfo->line, cu_gc_base(ptr));
    else
	fprintf(stderr, "No debug info for memory at %p.\n", ptr);
}

cuP_dbginfo_t
cuP_dbginfo(void *ptr)
{
#ifdef CUCONF_DEBUG_MEMORY
    void *base;
    if (enable_dbginfo)
	return GC_base(ptr);
#endif
    return NULL;
}

void
cu_debug_set_cmeminfo_D(void *ptr, void *client_key,
			cu_debug_cmeminfo_t cminfo)
{
    cuP_dbginfo_t hdr = cuP_dbginfo(ptr);
    if (hdr == NULL)
	return;
    cminfo->client_key = client_key;
    cminfo->chain = hdr->cminfo_chain;
    hdr->cminfo_chain = cminfo;
}

cu_debug_cmeminfo_t
cu_debug_cmeminfo_D(void *ptr, void *client_key)
{
    cuP_dbginfo_t hdr = cuP_dbginfo(ptr);
    cu_debug_cmeminfo_t cminfo;
    if (!hdr)
	return NULL;
    cu_debug_assert(ptr != NULL);
    for (cminfo = hdr->cminfo_chain; cminfo; cminfo = cminfo->chain)
	if (cminfo->client_key == client_key)
	    break;
    return cminfo;
}

#ifdef CUCONF_DEBUG_MEMORY
struct cuP_debug_finaliser_cd_s
{
    GC_finalization_proc fn;
    void *cd;
};
static void
cuP_debug_finaliser(void *obj, void *cd)
{
    obj += CU_ALIGNED_SIZEOF(struct cuP_dbginfo_s);
    assert(obj == cu_gc_base(obj));
    (*((struct cuP_debug_finaliser_cd_s *)cd)->fn)(
	obj,
	((struct cuP_debug_finaliser_cd_s *)cd)->cd);
}
void
cuD_gc_register_finaliser(void *ptr, GC_finalization_proc fn, void *cd,
			       GC_finalization_proc *ofn, void **ocd)
{
#ifndef CU_NDEBUG
    if (ptr != cu_gc_base(ptr)) {
	cu_errf("Non-base pointer %p vs %p passed for registration "
		 "of finaliser.", ptr, cu_gc_base(ptr));
	cu_debug_abort();
    }
#endif
    if (enable_dbginfo) {
	ptr -= CU_ALIGNED_SIZEOF(struct cuP_dbginfo_s);
	if (fn) {
	    struct cuP_debug_finaliser_cd_s *newcd
		= cu_gnew(struct cuP_debug_finaliser_cd_s);
	    newcd->fn = fn;
	    newcd->cd = cd;
	    GC_register_finalizer(ptr, &cuP_debug_finaliser, newcd, ofn, ocd);
	}
	else
	    GC_register_finalizer(ptr, NULL, NULL, ofn, ocd);
	if (ofn && *ofn == cuP_debug_finaliser) {
	    *ofn = ((struct cuP_debug_finaliser_cd_s *)*ocd)->fn;
	    *ocd = ((struct cuP_debug_finaliser_cd_s *)*ocd)->cd;
	}
    }
    else
	GC_register_finalizer(ptr, fn, cd, ofn, ocd);
}
void
cuD_gc_register_finaliser_no_order(void *ptr,
				       GC_finalization_proc fn, void *cd,
				       GC_finalization_proc *ofn, void **ocd)
{
#ifndef CU_NDEBUG
    if (ptr != cu_gc_base(ptr)) {
	cu_errf("Non-base pointer %p vs %p passed for registration "
		 "of finaliser.", ptr, cu_gc_base(ptr));
	cu_debug_abort();
    }
#endif
    if (enable_dbginfo) {
	ptr -= CU_ALIGNED_SIZEOF(struct cuP_dbginfo_s);
	if (fn) {
	    struct cuP_debug_finaliser_cd_s *newcd
		= cu_gnew(struct cuP_debug_finaliser_cd_s);
	    newcd->fn = fn;
	    newcd->cd = cd;
	    GC_register_finalizer_no_order(ptr, &cuP_debug_finaliser,
					   newcd, ofn, ocd);
	}
	else
	    GC_register_finalizer_no_order(ptr, NULL, NULL, ofn, ocd);
	if (ofn && *ofn == cuP_debug_finaliser) {
	    *ofn = ((struct cuP_debug_finaliser_cd_s *)*ocd)->fn;
	    *ocd = ((struct cuP_debug_finaliser_cd_s *)*ocd)->cd;
	}
    }
    else
	GC_register_finalizer_no_order(ptr, fn, cd, ofn, ocd);
}
#endif /* CUCONF_DEBUG_MEMORY */


/* Initialisation
 * ============== */

void
cuP_memory_init()
{
    void *ptr;

    cu_debug_assert(!memory_init_done);
    memory_init_done = 1;

    /* Determine size of debug header. */
    ptr = cu_galloc(1);
    cuD_gc_base_shift = (char*)ptr - (char*)GC_base(ptr);
    cu_gfree(ptr);
}

void
cuP_memory_init_debug()
{
    atexit(cuP_debug_show_all_segments);
}

void
cuP_memory_init_nodebug()
{
}
