/*-*-C-*-
 * Copyright 2006  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CU_SCRATCH_H
#define CU_SCRATCH_H

#include <cu/fwd.h>

CU_BEGIN_DECLARATIONS
/*!\defgroup cu_scratch_h cu/scratch.h: Scratch Memory for Repeated Reuse
 *@{\ingroup cu_mod
 * This header defines a small object to manage reuse of a scratch memory
 * area.  It is mainly inteded to be defined in function bodies and used in
 * inner loops when \ref cu_salloc is not feasible.
 */

typedef struct cu_scratch_s *cu_scratch_t;

/*!Reusable scratch memory. */
struct cu_scratch_s
{
    size_t size;
    void *ptr;
};

/*!Initialiser which can be assigned to a variable of type <tt>struct \ref
 * cu_scratch_s</tt>. */
#define CU_SCRATCH_INIT { 0, NULL }

void cuP_scratch_resize_min(cu_scratch_t, size_t);
void cuP_scratch_resize_log(cu_scratch_t, size_t);

/*!Extend \a scr to \a size unless it already is at least that big, and
 * return a pointer to it's memory.  Does not preserve the contents. */
CU_SINLINE void *
cu_scratch_alloc_min(cu_scratch_t scr, size_t size)
{
    if (scr->size < size)
	cuP_scratch_resize_min(scr, size);
    return scr->ptr;
}

/*!Extend \a scr to the maximum of \a size and twice the old size, unless the
 * old size is at least \a size, and return a pointer to it's memory.  Does
 * not preserve the contents. */
CU_SINLINE void *
cu_scratch_alloc_log(cu_scratch_t scr, size_t size)
{
    if (scr->size < size)
	cuP_scratch_resize_log(scr, size);
    return scr->ptr;
}

/*!@}*/
CU_END_DECLARATIONS

#endif
