/*-*-C-*-
 * Copyright 2002--2004  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CU_SREF_H
#define CU_SREF_H

#include <stdio.h>
#include <cu/fwd.h>

CU_BEGIN_DECLARATIONS
/*!\defgroup cu_sref cu/sref.h: Source Code References
 * @{ \ingroup cu_mod */

struct cu_sref_s
{
    cu_sref_t chain_tail;
    cu_str_t path;
    int line;
    int column;
    int tabstop;
};

/*!Construct a source file reference.  <tt>\a path = NULL</tt> means stdin.
 * If column numbers are not used, pass <tt>\a column = -1</tt>. */
void cu_sref_cct(cu_sref_t, cu_str_t path, int line, int column);
/*!\copydoc cu_sref_cct */
cu_sref_t cu_sref_new(cu_str_t path, int line, int column);
/*!\copydoc cu_sref_cct */
cu_sref_t cu_sref_new_cstr(char const *path, int line, int column);

/*!Modify \a new_head to be the head of a chain of source references
 * which continues with \a old_chain. */
CU_SINLINE void
cu_sref_set_chain_tail(cu_sref_t head, cu_sref_t chain_tail)
{ head->chain_tail = chain_tail; }

/*!The next in a chain of references. */
CU_SINLINE cu_sref_t
cu_sref_chain_tail(cu_sref_t srf) { return srf->chain_tail; }

/*!Change the width of tabulator stops. The default is 8. */
void cu_sref_set_tabstop(cu_sref_t, int w);

/*!Return an unknown source reference. */
CU_SINLINE cu_sref_t
cu_sref_unknown(void) { return NULL; }

/*!Return non-zero iff \a srf is known. */
CU_SINLINE int
cu_sref_is_known(cu_sref_t srf) { return srf != NULL; }

/*!Construct \a srf as a copy af \a srf0.  Chained references will
 * be shared, so make sure you only modify \a srf itself. */
void cu_sref_cct_copy(cu_sref_t srf, cu_sref_t srf0);

/*!Return a copy of \a srf0. */
cu_sref_t cu_sref_new_copy(cu_sref_t);

/* Return non-zero iff \a srf is defined. */
/* int cu_sref_is_def(cu_sref_t srf); */
/* #define cu_sref_is_def(srf) ((srf)->line >= 0) */

/*!Return the path of \a srf. */
CU_SINLINE cu_str_t
cu_sref_path(cu_sref_t srf) { return srf->path; }

/*!Return non-zero iff \a srf refers to standard input. */
CU_SINLINE int
cu_sref_is_stdin(cu_sref_t srf) { return srf->path == NULL; }

/*!Return the line number of \a srf. */
CU_SINLINE int
cu_sref_line(cu_sref_t srf) { return srf->line; }

/*!Return the column number of \a srf. */
CU_SINLINE int
cu_sref_column(cu_sref_t srf) { return srf->column; }

/*!Advance the column number by 1. */
CU_SINLINE void
cu_sref_next(cu_sref_t srf) { ++srf->column; }

/*!Advance the column number to that after a tab at the current
 * column. */
void cu_sref_tab(cu_sref_t);

/*!Advance the line number by 1 and reset the column number to 0. */
void cu_sref_newline(cu_sref_t);

/*!Print the source reference \a file in the format file_name:line
 * or, if column is defined, file_name:line:column. */
void cu_sref_fprint(cu_sref_t, FILE* file);

/*!@}*/
CU_END_DECLARATIONS
#endif
