#include <cucon/layout.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <gc/gc_pthread_redirects.h>
#include <time.h>

#define THREAD_CNT 5
#define TEST_CNT 50000

void
show_live_data_size(cucon_layout_t lo)
{
    size_t size = 0;
    while (lo) {
	size += sizeof(struct cucon_layout_s) + sizeof(cuex_meta_t);
	lo = lo->prefix;
    }
    printf("Approx. live data size = %ld\n", (long)size);
}

cucon_layout_t
show_grow(cucon_layout_t l, size_t bitsize, size_t bitalign)
{
    cu_offset_t bitoffset;
    cucon_layout_t lr = cucon_layout_pack_bits(l, bitsize, bitalign, &bitoffset);
    printf("grow(%p, %ld, %ld, => %ld) = %p\n",
	    l, (long)bitsize, (long)bitalign, (long)bitoffset, lr);
    return lr;
}

cucon_layout_t
show_join(cucon_layout_t l0, cucon_layout_t l1)
{
    cu_offset_t bitoffset;
    cucon_layout_t lr = cucon_layout_product(l0, l1, &bitoffset);
    printf("join(%p, %p, => %ld) = %p\n", l0, l1, (long)bitoffset, lr);
    return lr;
}

void
simple_test()
{
    cucon_layout_t l0, l1;
    l0 = show_grow(NULL, 3, 1);
    l0 = show_grow(l0, 4, 1);
    l0 = show_grow(l0, sizeof(long)*8, sizeof(long)*8);
    l1 = show_join(l0, l0);
    l1 = show_join(l0, l1);
}

void
union_test()
{
    cucon_layout_t l0;
    cucon_layout_t l1;
    printf("\nUnion Test\n\n");
    l0 = show_grow(NULL, 2, 1);
    l0 = show_grow(l0, 8, 8);
    l0 = show_grow(l0, 32, 32);
    l0 = show_grow(l0, 16, 8);
    cucon_layout_dump(l0, stdout);
    l1 = show_grow(NULL, 1, 1);
    l1 = show_grow(l1, 16, 8);
    cucon_layout_dump(l1, stdout);

    printf("Taking union\n");
    l0 = cucon_layout_union(l0, l1);
    cucon_layout_dump(l0, stdout);
    l0 = show_grow(l0, 1, 1);
    cucon_layout_dump(l0, stdout);
}

void
product_test()
{
    cu_offset_t offset;
    cucon_layout_t l0, l1, l2;
    l0 = show_grow(NULL, 3, 1);
    l0 = show_grow(l0, 8, 8);
    l1 = show_grow(NULL, 2, 1);
    printf("Product of\n");
    cucon_layout_dump(l0, stdout);
    printf("and\n");
    cucon_layout_dump(l1, stdout);
    l2 = cucon_layout_product(l0, l1, &offset);
    printf("gives offset %ld in\n", (long int)offset);
    cucon_layout_dump(l2, stdout);
}

void *
test_layout(void *data)
{
    int I = (uintptr_t)data;
    int i;
    cucon_layout_t lo = 0;
    for (i = 0; i < I; ++i) {
	cu_offset_t bitsize = lrand48() % (sizeof(long)*12) + 1;
	cu_offset_t bitalign;
	cu_offset_t offset;
	if (i % (I*THREAD_CNT/20) == 0)
	    printf("%4.2lf complete\n", i/(double)I);
	//if (i % (I/100) == 0)
	//    GC_gcollect();
	if (bitsize > sizeof(long)*8) {
	    switch (bitsize % 8) {
	    case 0:
		bitalign = bitsize = sizeof(char)*8;
		break;
	    case 1:
		bitalign = bitsize = sizeof(short)*8;
		break;
	    case 2:
		bitalign = bitsize = sizeof(int)*8;
		break;
	    case 3:
		bitalign = bitsize = sizeof(long)*8;
		break;
	    default:
		if (lo)
		    lo = cucon_layout_prefix(lo);
		if (lo)
		    lo = cucon_layout_prefix(lo);
		goto next_i;
	    }
	}
	else
	    bitalign = 1;
	cu_debug_assert(bitsize > 0);
	lo = cucon_layout_pack_bits(lo, bitsize, bitalign, &offset);
	//if (I < 400)
	//    printf("%2d %2d %4d\n", bitsize, bitalign, offset);
    next_i:;
    }
    show_live_data_size(lo);
    return NULL;
}

long test_seed;

int
main()
{
    pthread_t th[THREAD_CNT];
    int i;
    cu_init();
    test_seed = time(NULL);
    //test_seed = 1121870991;
    srand48(test_seed);
    //test_layout((void *)100);
    simple_test();
    union_test();
    product_test();
#if THREAD_CNT <= 1
    test_layout((void *)(TEST_CNT/THREAD_CNT));
#else
    for (i = 0; i < THREAD_CNT; ++i) {
	int err;
	err = GC_pthread_create(&th[i], NULL, test_layout,
				(void *)(TEST_CNT/THREAD_CNT));
	if (err) {
	    fprintf(stderr, "%s\n", strerror(err));
	    return 1;
	}
    }
    for (i = 0; i < THREAD_CNT; ++i)
	GC_pthread_join(th[i], NULL);
#endif
    return 0;
}
