/*-*-C-*-
 * Copyright 2005  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cudyn/type.h>
#include <cudyn/misc.h>
#include <cucon/stack.h>
#include <cuex/opr.h>
#include <cuex/aci.h>
#include <cuex/algo.h>
#include <cuex/sig.h>
#include <cuex/type.h>


static cu_mutex_t type_glck = CU_MUTEX_INITIALISER;


/* Pointer Types
 * ============= */

/* May consider to add the deref type to the structure, but this means it must
 * be computed in case of cudyn_ptrtype_from_ex. */

cudyn_ptrtype_t
cudyn_ptrtype_from_ex(cuex_t ex)
{
    struct cudyn_ptrtype_s init;
    /*cu_debug_assert(cuex_meta(ex) == cuex_opr_ptr_to_1);*/
    cudynP_hctype_cct_hcs(cu_to2(cudyn_hctype, cudyn_inltype, &init), ex,
			  cudyn_typekind_ptrtype, sizeof(void *));
    cu_to(cudyn_inltype, &init)->layout = (AO_t)cucon_layout_ptr;
    cu_to(cudyn_inltype, &init)->ffitype = (AO_t)&ffi_type_pointer;
    return cudyn_hnew(cudyn_ptrtype, &init);
}

cudyn_ptrtype_t
cudyn_ptrtype(cuex_t deref)
{
    struct cudyn_ptrtype_s init;
    cuex_t ex = cuex_o1(ptr_to, cudyn_type_as_expr(deref));
    cudynP_hctype_cct_hcs(cu_to2(cudyn_hctype, cudyn_inltype, &init), ex,
			  cudyn_typekind_ptrtype, sizeof(void *));
    cu_to(cudyn_inltype, &init)->layout = (AO_t)cucon_layout_ptr;
    cu_to(cudyn_inltype, &init)->ffitype = (AO_t)&ffi_type_pointer;
    return cudyn_hnew(cudyn_ptrtype, &init);
}


/* Elementary Types
 * ================ */

cudyn_elmtype_t
cudyn_elmtype_new(cudyn_typekind_t kind,
		  cu_offset_t size, cu_offset_t alignment, ffi_type *ffitype)
{
    cu_offset_t bitoffset;
    cudyn_elmtype_t t = cudyn_onew(cudyn_elmtype);
    cu_offset_t wsize;
    wsize = (size + sizeof(cu_word_t) - 1)/sizeof(cu_word_t)*sizeof(cu_word_t);
    cudynP_hctype_cct_hcs(cu_to2(cudyn_hctype, cudyn_inltype, t), NULL,
			  kind, wsize);
    cu_to(cudyn_inltype, t)->layout
	= (AO_t)cucon_layout_pack_bits(NULL, size*8, alignment*8, &bitoffset);
    cu_to(cudyn_inltype, t)->ffitype = (AO_t)ffitype;
    return t;
}


/* Array Types
 * =========== */

static void
arrtype_cct_glck(cudyn_arrtype_t t, cuex_t ex)
{
    cu_offset_t bitoffset;
    size_t elt_bitsize, elt_bitalign;
    size_t arr_bitsize, arr_bitalign;
    cucon_layout_t lyo, sub_lyo;
    cu_debug_assert(cuex_meta(ex) == cuex_opr_gexpt_2);
    if (!cudyn_is_int(cuex_opn_at(ex, 1)))
	return;
    t->elt_type = cudyn_type_glck(cuex_opn_at(ex, 0));
    if (!t->elt_type)
	return;
    t->elt_cnt = cudyn_to_int(cuex_opn_at(ex, 1));
    sub_lyo = cudyn_type_layout(t->elt_type);
    elt_bitsize = cucon_layout_bitsize(sub_lyo);
    elt_bitalign = cucon_layout_bitalign(sub_lyo);
    arr_bitsize = elt_bitsize*t->elt_cnt;
    arr_bitalign = elt_bitalign;
    lyo = cucon_layout_pack_bits(NULL, arr_bitsize, arr_bitalign, &bitoffset);
    cudynP_hctype_cct_hcs(cu_to2(cudyn_hctype, cudyn_inltype, t), ex,
			  cudyn_typekind_arrtype, cucon_layout_size(lyo));
    AO_store_release_write(&cu_to(cudyn_inltype, t)->layout, (AO_t)lyo);
}

cudyn_arrtype_t
cudyn_arrtype_glck(cuex_t ex)
{
    cudyn_arrtype_t t;
    t = cudyn_hnew_general(cudyn_arrtype, sizeof(cuex_t), &ex);
    if (!AO_load_acquire_read(&cu_to(cudyn_inltype, t)->layout))
	arrtype_cct_glck(t, ex);
    if (!cu_to(cudyn_inltype, t)->layout)
	return NULL;
    return t;
}

cudyn_arrtype_t
cudyn_arrtype(cudyn_type_t elt_type, size_t cnt)
{
    cudyn_arrtype_t t;
    cu_mutex_lock(&type_glck);
    t = cudyn_arrtype_glck(cuex_o2(gexpt, elt_type, cudyn_int(cnt)));
    cu_mutex_unlock(&type_glck);
    return t;
}


/* Product Types
 * ============= */

static cucon_layout_t
tuptype_finish_gprod_glck(cudyn_tuptype_t t, cuex_t ex, int i)
{
    cudyn_type_t subt;
    cucon_layout_t lyo;
    if (cuex_meta(ex) == cuex_opr_gprod_2) {
	cu_debug_assert(i > 0);
	subt = cudyn_type_glck(cuex_opn_at(ex, 1));
	if (!subt)
	    return NULL;
	lyo = tuptype_finish_gprod_glck(t, cuex_opn_at(ex, 0), i - 1);
	if (!lyo)
	    return NULL;
	t->tcomp_arr[i].type = subt;
	return cucon_layout_product(lyo, cudyn_type_layout(subt),
				    &t->tcomp_arr[i].bitoffset);
    }
    else {
	cu_debug_assert(i == 0);
	subt = cudyn_type_glck(ex);
	if (!subt)
	    return NULL;
	t->tcomp_arr[0].type = subt;
	t->tcomp_arr[0].bitoffset = 0;
	return cudyn_type_layout(subt);
    }
}

cu_clos_def(tuptype_finish_sigprod_cb,
	    cu_bool_t cu_proto(cuex_opn_t e),
	( cucon_layout_t lyo;
	  cudyn_tuptype_t t; ))
{
    cu_clos_self(tuptype_finish_sigprod_cb);
    cudyn_type_t subt;
    cu_idr_t idr = cuex_aci_at(e, 0);
    struct cudyn_tupcomp_s *comp;
    if (!cucon_pmap_insert_mem(&self->t->scomp_map, idr,
			       sizeof(struct cudyn_tupcomp_s), &comp))
	cu_debug_unreachable();
    subt = cudyn_type_glck(cuex_aci_at(e, 1));
    if (!subt)
	return cu_false;
    comp->type = subt;
    self->lyo = cucon_layout_product(self->lyo, cudyn_type_layout(subt),
				     &comp->bitoffset);
    return cu_true;
}

static cucon_layout_t
tuptype_finish_sigprod_glck(cudyn_tuptype_t t, cuex_t ex, cucon_layout_t lyo)
{
    tuptype_finish_sigprod_cb_t cb;
    cu_clos_cct(&cb, tuptype_finish_sigprod_cb);
    cb.lyo = lyo;
    cb.t = t;
    if (cuex_aci_conj(cuex_opr_sigprod_2aci2, ex, cu_clos_ref(&cb)))
	return cb.lyo;
    else
	return NULL;
}

void
tuptype_cct_glck(cudyn_tuptype_t t, cuex_t ex)
{
    cucon_layout_t lyo;
    cucon_pmap_cct(&t->scomp_map);
    if (ex == cuex_sig_identity()) {
	lyo = 0;
	t->tcomp_cnt = 0;
	t->tcomp_arr = NULL;
    }
    else if (cuex_meta(ex) == cuex_opr_gprod_2) {
	cuex_t ex0 = cuex_opn_at(ex, 0);
	cuex_t ex1 = cuex_opn_at(ex, 1);
	if (cuex_meta(ex1) == cuex_opr_sigprod_2aci2) {
	    t->tcomp_cnt = cuex_binary_left_depth(cuex_opr_gprod_2, ex0) + 1;
	    t->tcomp_arr =
		cu_galloc(t->tcomp_cnt*sizeof(struct cudyn_tupcomp_s));
	    lyo = tuptype_finish_gprod_glck(t, ex0, t->tcomp_cnt - 1);
	    if (!lyo)
		return;
	    lyo = tuptype_finish_sigprod_glck(t, ex1, lyo);
	    if (!lyo)
		return;
	}
	else {
	    t->tcomp_cnt = cuex_binary_left_depth(cuex_opr_gprod_2, ex) + 1;
	    t->tcomp_arr =
		cu_galloc(t->tcomp_cnt*sizeof(struct cudyn_tupcomp_s));
	    lyo = tuptype_finish_gprod_glck(t, ex, t->tcomp_cnt - 1);
	    if (!lyo)
		return;
	}
    }
    else if (cuex_meta(ex) == cuex_opr_sigprod_2aci2) {
	t->tcomp_cnt = 0;
	t->tcomp_arr = NULL;
	lyo = tuptype_finish_sigprod_glck(t, ex, NULL);
	if (!lyo)
	    return;
    }
    else {
	cudyn_type_t t0 = cudyn_type_glck(ex);
	if (!t0)
	    return;
	t->tcomp_cnt = 1;
	t->tcomp_arr = cu_galloc(sizeof(struct cudyn_tupcomp_s));
	t->tcomp_arr[0].type = t0;
	t->tcomp_arr[0].bitoffset = 0;
	lyo = cudyn_type_layout(t0);
	cu_debug_assert(lyo);
    }
    cudynP_hctype_cct_hcs(cu_to2(cudyn_hctype, cudyn_inltype, t), ex,
			  cudyn_typekind_tuptype, cucon_layout_size(lyo));
    AO_store_release_write(&cu_to(cudyn_inltype, t)->layout, (AO_t)lyo);
}

cudyn_tuptype_t
cudyn_tuptype_glck(cuex_t ex)
{
    cudyn_tuptype_t t;
    t = cudyn_hnew_general(cudyn_tuptype, sizeof(cuex_t), &ex);
    if (!cu_to(cudyn_inltype, t)->layout)
	tuptype_cct_glck(t, ex);
    if (!cu_to(cudyn_inltype, t)->layout)
	return NULL;
    return t;
}

cudyn_tuptype_t
cudyn_tuptype(cuex_t ex)
{
    cudyn_tuptype_t t;
    t = cudyn_hnew_general(cudyn_tuptype, sizeof(cuex_t), &ex);
    if (!AO_load_acquire_read(&cu_to(cudyn_inltype, t)->layout)) {
	cu_mutex_lock(&type_glck);
	if (!cu_to(cudyn_inltype, t)->layout)
	    tuptype_cct_glck(t, ex);
	cu_mutex_unlock(&type_glck);
	if (!cu_to(cudyn_inltype, t)->layout)
	    return NULL;
    }
    return t;
}

cudyn_tuptype_t
cudyn_tuptype_by_valist(cu_offset_t cnt, va_list vl)
{
    cuex_t e;
    cu_debug_assert(cnt > 0);
    e = va_arg(vl, cuex_t);
    while (--cnt)
	e = cuex_o2(gprod, e, va_arg(vl, cuex_t));
    return cudyn_tuptype(e);
}

cu_clos_def(tuptype_conj_cb,
	    cu_bool_t cu_proto(void const *idr, void *slot),
	( cu_clop(cb, cu_bool_t, cu_idr_t, cu_offset_t, cudyn_type_t); ))
{
    cu_clos_self(tuptype_conj_cb);
    struct cudyn_tupcomp_s *comp = slot;
    return cu_call(self->cb, (cu_idr_t)idr, comp->bitoffset, comp->type);
}

cu_bool_t
cudyn_tuptype_conj(cudyn_tuptype_t t,
		   cu_clop(cb, cu_bool_t, cu_idr_t, cu_offset_t, cudyn_type_t))
{
    tuptype_conj_cb_t scb;
    size_t i;
    struct cudyn_tupcomp_s *comp = t->tcomp_arr;
    for (i = 0; i < t->tcomp_cnt; ++i) {
	if (!cu_call(cb, NULL, comp->bitoffset, comp->type))
	    return cu_false;
	++comp;
    }
    cu_clos_cct(&scb, tuptype_conj_cb);
    scb.cb = cb;
    return cucon_pmap_conj_mem(&t->scomp_map, cu_clos_ref(&scb));
}


/* Union Types
 * =========== */

cu_clos_def(duntype_cct_cb,
	    cu_bool_t cu_proto(cuex_opn_t node),
	( cucon_layout_t lyo;
	  cudyn_cnum_t cnum;
	  cudyn_duntype_t t; ))
{
    cu_clos_self(duntype_cct_cb);
    cuex_t typeex;
    struct cudyn_dunpart_s *part;
    if (!cucon_pmap_insert_mem(&self->t->idr_to_part, cuex_aci_at(node, 0),
			       sizeof(struct cudyn_dunpart_s), &part))
	cu_debug_unreachable();
    typeex = cuex_binary_inject_left(cuex_opr_gprod_2, cuex_aci_at(node, 1),
				     cudyn_int_type());
    part->cnum = self->cnum++;
    part->type = cudyn_type_glck(typeex);
    if (!part->type)
	return cu_false;
    self->lyo = cucon_layout_union(self->lyo, cudyn_type_layout(part->type));
    return cu_true;
}

static void
duntype_cct_glck(cudyn_duntype_t t, cuex_t ex)
{
    duntype_cct_cb_t cb;
    cu_debug_assert(cuex_meta(ex) == cuex_opr_dunion_2aci2);
    cucon_pmap_cct(&t->idr_to_part);
    cu_clos_cct(&cb, duntype_cct_cb);
    cb.lyo = NULL;
    cb.t = t;
    cb.cnum = 0;
    if (!cuex_aci_conj(cuex_opr_dunion_2aci2, ex, cu_clos_ref(&cb)))
	return;
    /* TODO. Hash cons option, variable size. */
    cudynP_hctype_cct_nonhc(cu_to2(cudyn_hctype, cudyn_inltype, t),
			    ex, cudyn_typekind_duntype);
    AO_store_release_write(&cu_to(cudyn_inltype, t)->layout, (AO_t)cb.lyo);
}

cudyn_duntype_t
cudyn_duntype_glck(cuex_t ex)
{
    cudyn_duntype_t t;
    t = cudyn_hnew_general(cudyn_duntype, sizeof(cuex_t), &ex);
    if (!cu_to(cudyn_inltype, t)->layout)
	duntype_cct_glck(t, ex);
    if (!cu_to(cudyn_inltype, t)->layout)
	return NULL;
    return t;
}


/* Singular Types
 * ============== */

cudyn_sngtype_t
cudyn_sngtype_glck(cuex_t ex)
{
    cudyn_sngtype_t t;
    t = cudyn_hnew_general(cudyn_sngtype, sizeof(cuex_t), &ex);
    if (!cu_to(cudyn_inltype, t)->ffitype) {
	cudynP_hctype_cct_nonhc(cu_to2(cudyn_hctype, cudyn_inltype, t),
				ex, cudyn_typekind_sngtype);
	AO_store_release_write(&cu_to(cudyn_inltype, t)->ffitype,
			       (AO_t)&ffi_type_void);
    }
    return t;
}

cudyn_sngtype_t
cudyn_sngtype(cuex_t ex)
{
    cudyn_sngtype_t t;
    t = cudyn_hnew_general(cudyn_sngtype, sizeof(cuex_t), &ex);
    if (!AO_load_acquire_read(&cu_to(cudyn_inltype, t)->ffitype)) {
	cu_mutex_lock(&type_glck);
	if (!cu_to(cudyn_inltype, t)->ffitype) {
	    cudynP_hctype_cct_nonhc(cu_to2(cudyn_hctype, cudyn_inltype, t),
				    ex, cudyn_typekind_sngtype);
	    AO_store_release_write(&cu_to(cudyn_inltype, t)->ffitype,
				   (AO_t)&ffi_type_void);
	}
	cu_mutex_unlock(&type_glck);
    }
    return t;
}

cudyn_sngtype_t
cudyn_sngtype_of_elt(cuex_t elt)
{
    return cudyn_sngtype(cuex_aci_generator(cuex_opr_setjoin_2aci1, elt));
}


/* Generic
 * ======= */

cudyn_type_t
cudyn_type_glck(cuex_t ex)
{
    if (cudyn_is_type(ex))
	return ex;
    switch (cuex_meta(ex)) {
	case cuex_opr_setjoin_2aci1:
	    return cudyn_sngtype_to_type(cudyn_sngtype_glck(ex));
	case cuex_opr_gexpt_2:
	    return cudyn_arrtype_to_type(cudyn_arrtype_glck(ex));
	case cuex_opr_gprod_2:
//	case cuex_opr_sigprod_2aci2:
	    return cudyn_tuptype_to_type(cudyn_tuptype_glck(ex));
	case cuex_opr_dunion_2aci2:
	    return cudyn_duntype_to_type(cudyn_duntype_glck(ex));
	case cuex_opr_farrow_2:
	case cuex_opr_farrow_native_2:
	    /* TODO, for now. */
	    return cudyn_ptrtype_to_type(cudyn_ptrtype_from_ex(ex));
	default:
#if 0
	    cu_bugf("Invalid or unimplemented type expression.");
	    return NULL;
#else
	    {
		struct cudyn_hctype_s tpl;
		cudynP_hctype_cct_hcs(&tpl, ex, cudyn_typekind_by_expr,
				      cuex_type_size(ex));
		return cudyn_halloc_general(cudyn_type_type(),
			    sizeof(struct cudyn_hctype_s),
			    sizeof(struct cudyn_hctype_s) - CU_HCOBJ_SHIFT,
			    sizeof(cuex_t),
			    cu_ptr_add(&tpl, CU_HCOBJ_SHIFT));
	    }
#endif
    }
}

cudyn_type_t
cudyn_type(cuex_t ex)
{
    cudyn_type_t t;
    cu_mutex_lock(&type_glck);
    t = cudyn_type_glck(ex);
    cu_mutex_unlock(&type_glck);
    return t;
}


/* Init
 * ==== */

cudyn_stdtype_t cudynP_cuex_type;
cudyn_stdtype_t cudynP_ptrtype_type;
cudyn_stdtype_t cudynP_elmtype_type;
cudyn_stdtype_t cudynP_arrtype_type;
cudyn_stdtype_t cudynP_tuptype_type;
cudyn_stdtype_t cudynP_sngtype_type;
cudyn_stdtype_t cudynP_duntype_type;
cuex_t cudynP_tup_null;
cudyn_sngtype_t cudynP_default_sngtype;
cudyn_stdtype_t cudynP_type_type;

void
cudynP_type_init()
{
    cudynP_cuex_type = cudyn_stdtype_new();
    cudynP_ptrtype_type = cudyn_stdtypeoftypes_new_hcs(
	    sizeof(struct cudyn_ptrtype_s) - CU_HCOBJ_SHIFT);
    cudynP_elmtype_type = cudyn_stdtypeoftypes_new();
    cudynP_arrtype_type = cudyn_stdtypeoftypes_new_hce();
    cudynP_tuptype_type = cudyn_stdtypeoftypes_new_hce();
    cudynP_duntype_type = cudyn_stdtypeoftypes_new_hce();
    cudynP_sngtype_type = cudyn_stdtypeoftypes_new_hce();
    cudynP_tup_null = cuex_aci_identity(cuex_opr_sigprod_2aci2);
    cudynP_default_sngtype = cudyn_sngtype_of_elt(cudynP_tup_null);
    cudynP_type_type = cudyn_stdtypeoftypes_new_hce();
}
