/*-*-C-*-
 * Copyright 2005  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cuex/fwd.h>
#include <cugra/fwd.h>
#include <cuex/var.h>
#include <cuex/tpvar.h>
#include <cuex/tvar.h>
#include <cuex/opn.h>
#include <cuex/oprinfo.h>
#include <cuex/dunion.h>
#include <cudyn/properties.h>
#include <cucon/pmap.h>
#include <cu/va_ref.h>

/* This printing is meant for debugging purposes.  It will lock all printed
 * variables from being GCed.  That could be avoided by using cucon_wmap_t
 * instead of cucon_pmap_t in varindex, but then two variables which prints
 * the same may in fact be different. */

static int
varindex(cuex_t var)
{
    static cucon_pmap_t index_map = NULL;
    static int current_index[] = {0, 0, 0, 0};
    static cu_mutex_t mutex = CU_MUTEX_INITIALISER;
    int *i;
    cuex_qcode_t qc = cuex_varmeta_qcode(cuex_meta(var));
    cu_debug_assert(qc < 4);
    cu_mutex_lock(&mutex);
    if (!index_map)
	index_map = cucon_pmap_new();
    if (cucon_pmap_insert_mem(index_map, var, sizeof(int), &i))
	*i = ++current_index[qc];
    cu_mutex_unlock(&mutex);
    return *i;
}

static void
cuex_print_ex(cuex_t ex, FILE *out)
{
    cuex_meta_t meta = cuex_meta(ex);
    switch (cuex_meta_kind(meta)) {
	    cuex_oprinfo_t oi;
	    int i, r;
	case cuex_meta_kind_opr:
	    oi = cuex_oprinfo(meta);
	    if (oi)
		fprintf(out, "(#%s", cuex_oprinfo_name(oi));
	    else
		fprintf(out, "(#%lx", (unsigned long)meta);
	    r = cuex_opr_r(meta);
	    for (i = 0; i < r; ++i) {
		fputc(' ', out);
		cuex_print_ex(cuex_opn_at(ex, i), out);
	    }
	    fputc(')', out);
	    break;
	case cuex_meta_kind_type:
	    if (!cudyn_raw_print(ex, out))
		fprintf(out, "C%p", ex);
	    break;
	default:
	    if (cuex_is_varmeta(meta)) {
		if (cuex_is_tvarmeta(meta) && cuex_tvar_is_type(ex))
		    switch (cuex_varmeta_qcode(meta)) {
			case cuex_qcode_u:
			    fputs("α", out);
			    break;
			case cuex_qcode_e:
			    fputs("ξ", out);
			    break;
			default:
			    fputs("μ", out);
			    break;
		    }
		else
		    switch (cuex_varmeta_qcode(meta)) {
			case cuex_qcode_u:
			    fputc('u', out);
			    break;
			case cuex_qcode_e:
			    fputc('e', out);
			    break;
			case cuex_qcode_w:
			    fputc('w', out);
			    break;
			case cuex_qcode_n:
			    fputc('n', out);
			    break;
		    }
		fprintf(out, "%d", varindex(ex));
	    }
	    else
		fprintf(out, "special_%p", ex);
    }
}

cu_clop_def(format_expression, void, cu_str_t fmt, cu_va_ref_t va, FILE *out)
{
    cuex_t ex = cu_va_ref_arg(va, cuex_t);
    cuex_print_ex(ex, out);
}


void cudynP_init(void);
void cuexP_ex_init(void);
void cuexP_misc_init(void);
void cuexP_sig_init(void);
void cuexP_type_init(void);
void cuex_opr_init(void);

cuex_opn_t cuexP_dunion_empty;

void
cuex_init()
{
    static int done_init = 0;
    if (done_init)
	return;
    done_init = 1;

    cu_debug_assert(sizeof(struct cuex_tpvar_s)/sizeof(cu_word_t) <=
		    (1 << CUEXP_VARMETA_WSIZE_WIDTH));

    cugra_init();
    cudynP_init();
    cuexP_ex_init();
    cuex_opr_init();
    cuexP_misc_init();
    cuexP_sig_init();
    cuexP_type_init();
    cu_diag_define_format_key('!', format_expression);
    cuexP_dunion_empty = cuex_aci_identity(cuex_opr_dunion_2aci2);
}
