/*-*-C-*-
 * Copyright 2006  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CUEX_TYPE_H
#define CUEX_TYPE_H

#include <cuex/fwd.h>
#include <cuex/opr.h>
#include <cudyn/type.h>

CU_BEGIN_DECLARATIONS
/*!\defgroup cuex_type_h cuex/type.h: Dynamic Type Support
 * @{\ingroup cuex_mod */

size_t cuex_type_bitsize(cuex_t type);

CU_SINLINE size_t cuex_type_size(cuex_t type)
{
    size_t s = cuex_type_bitsize(type);
    return (s + 7)/8;
}

cu_offset_t cuex_type_bitalign(cuex_t type);

CU_SINLINE cu_offset_t cuex_type_align(cuex_t type)
{
    cu_offset_t a = cuex_type_bitalign(type);
    return (a + 7)/8;
}

cu_rank_t cuex_farrow_arity(cuex_t type);

cuex_t cuex_typeof(cuex_t e);


/* Pointers
 * -------- */

extern cuex_t cuexP_generic_ptr_type;

/*!A generic pointer type. This is a \ref cuex_opr_ptr_to_1. */
CU_SINLINE cuex_t cuex_generic_ptr_type(void)
{ return cuexP_generic_ptr_type; }

/*!True iff \a e is \ref cuex_generic_ptr_type(). */
CU_SINLINE cu_bool_t cuex_is_generic_ptr_type(cuex_t e)
{ return e == cuexP_generic_ptr_type; }

/*!Returns \a fn as an expression of type \a t, which should be the type of \a
 * fn, expressed as a \ref cuex_opr_farrow_native_2 with the first operand
 * being the argument types as a left-associated tree of \ref
 * cuex_opr_gprod_2, and the second operand being the result type. E.g.
 * \code
 * cuex_t t = cuex_o2(farrow_native, cuex_o2(gprod, cudyn_double_type(), cudyn_double_type()), cudyn_double_type());
 * cuex_t e = cuex_of_fnptr(t, (cu_fnptr_t)atan2);
 * \endcode
 */
CU_SINLINE cuex_t
cuex_of_fnptr(cuex_t t, cu_fnptr_t fn)
{
    cu_debug_assert(cuex_meta(t) == cuex_opr_farrow_native_2
		    || cuex_meta(t) == cuex_opr_forall_2);
    return cudyn_halloc_by_key(cudyn_type(t), sizeof(void *), &fn);
}

CU_SINLINE cu_bool_t
cuex_is_fnptr(cuex_t t)
{
    cuex_meta_t m = cuex_meta(t);
    return cuex_meta_is_type(m)
	&& cuex_meta(cudyn_type_as_expr(cudyn_type_from_meta(m)))
	   == cuex_opr_farrow_native_2;
}

CU_SINLINE cu_fnptr_t
cuex_to_fnptr(cuex_t e)
{
    return *(cu_fnptr_t *)e;
}

/*!@}*/
CU_END_DECLARATIONS

#endif
