/*-*-C-*-
 * Copyright 2006  Petter Urkedal
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <cuflow/cache.h>
#include <cuflow/time.h>
#include <cu/wordarr.h>
#include <math.h>

#define FILL_MIN_NOM 1
#define FILL_MIN_DENOM 2
#define FILL_MAX_NOM 4
#define FILL_MAX_DENOM 3

#define CUFLOWP_CACHE_BIN_BITNO \
    (sizeof(cu_hash_t)*8 - CUFLOWP_CACHE_LOG2_BIN_COUNT)
#define HASH_BIN(hash) ((hash) >> CUFLOWP_CACHE_BIN_BITNO)
#define HASH_SLOT(hash, cap) ((hash) & ((1 << cap) - 1))
#define HDR CUFLOWP_CACHEOBJ_HDR
#define CUFLOW_CACHEOBJ_FNCODE(obj) ((obj)->fncode)
#define CACHEOBJ_KEY_SIZEW(obj) \
    CUFLOW_FNCODE_KEY_SIZEW(CUFLOW_CACHEOBJ_FNCODE(obj))

void
cuflow_cache_cct(cuflow_cache_t cache, cuflow_cacheconf_t conf,
		 cu_clop(*fn_arr, cuflow_cacheobj_t, cuflow_cacheobj_t key))
{
    int i;
    cache->conf = conf;
    cache->fn_arr = fn_arr;
    for (i = 0; i < CUFLOWP_CACHE_BIN_COUNT; ++i) {
	cuflowP_cachebin_t bin = &cache->bin_arr[i];
	cu_mutex_cct(&bin->mutex);
	bin->cap = 0;
	bin->size = 0;
	bin->link_arr = NULL;
    }
}

cuflow_cache_t
cuflow_cache_new(cuflow_cacheconf_t conf,
		 cu_clop(*fn_arr, cuflow_cacheobj_t, cuflow_cacheobj_t key))
{
    cuflow_cache_t cache = cu_gnew(struct cuflow_cache_s);
    cuflow_cache_cct(cache, conf, fn_arr);
    return cache;
}

cuflow_cacheobj_t
cuflow_cacheobj_new(cuflow_cacheobj_t key, size_t full_size)
{
    cuflowP_cacheobjhdr_t base;
    cuflow_cacheobj_t obj;
    size_t key_word_cnt = CACHEOBJ_KEY_SIZEW(key);
    full_size += sizeof(struct cuflowP_cacheobjhdr_s);
    base = cu_galloc(full_size);
    obj = (cuflow_cacheobj_t)(base + 1);
    cu_wordarr_copy((cu_word_t *)obj, (cu_word_t *)key, key_word_cnt);
    return obj;
}

CU_SINLINE cu_bool_t
cacheobj_eq(cuflow_cacheobj_t obj0, cuflow_cacheobj_t obj1)
{
    size_t word_cnt = CACHEOBJ_KEY_SIZEW(obj0);
    return cu_wordarr_eq((cu_word_t *)obj0, (cu_word_t *)obj1, word_cnt);
}

CU_SINLINE cu_hash_t
cacheobj_hash(cuflow_cacheobj_t obj)
{
    size_t word_cnt = CACHEOBJ_KEY_SIZEW(obj);
    return cu_wordarr_hash((cu_word_t *)obj + 1, word_cnt, *(cu_word_t *)obj);
}

static void
resize_lck(cuflowP_cachebin_t bin, size_t new_cap)
{
    cuflow_cacheobj_t *old_link_arr = bin->link_arr;
    cuflow_cacheobj_t *old_link_arr_end = old_link_arr + bin->cap;
    bin->cap = new_cap;
    bin->link_arr = cu_cgalloc(sizeof(cuflow_cacheobj_t)*new_cap);
    while (old_link_arr != old_link_arr_end) {
	cuflow_cacheobj_t obj = *old_link_arr;
	while (obj) {
	    cuflow_cacheobj_t next_obj = HDR(obj)->next;
	    cu_hash_t hash = cacheobj_hash(obj);
	    cuflow_cacheobj_t *slot;
	    slot = &bin->link_arr[HASH_SLOT(hash, new_cap)];
	    HDR(obj)->next = *slot;
	    *slot = obj;
	    obj = next_obj;
	}
	++old_link_arr;
    }
}

CU_SINLINE cu_bool_t
drop_condition(cuflow_cacheconf_t conf, cuflow_cacheobj_t obj)
{
    return conf->cost_per_cycle_per_byte
	 > conf->decay_per_walltime_unit
	   * HDR(obj)->access_function * HDR(obj)->gain;
}

CU_SINLINE void
update(cuflow_cacheconf_t conf, cuflow_cacheobj_t obj, float t)
{
    float minus_dt = HDR(obj)->t_access - t;
    minus_dt *= conf->decay_per_walltime_unit;
    HDR(obj)->t_access = t;
    HDR(obj)->access_function *= expf(minus_dt);
}

static void
prune_and_adjust_lck(cuflow_cache_t cache, cuflowP_cachebin_t bin)
{
    cuflow_cacheobj_t *link_arr = bin->link_arr;
    cuflow_cacheobj_t *link_arr_end = bin->link_arr + bin->cap;
    cuflow_cacheconf_t conf = cache->conf;
    float t = (float)cuflow_walltime();

    /* Go through bin and prune recently unused objects */
    while (link_arr != link_arr_end) {
	cuflow_cacheobj_t *obj_slot = link_arr;
	cuflow_cacheobj_t obj;
	while ((obj = *obj_slot)) {
	    update(conf, obj, t);
	    if (drop_condition(conf, obj))
		*obj_slot = HDR(obj)->next;
	    else
		obj_slot = &HDR(obj)->next;
	}
	++link_arr;
    }

    /* Adjust capacity if necessary. */
    if (bin->size*FILL_MAX_DENOM > bin->cap*FILL_MAX_NOM)
	resize_lck(bin, bin->cap*2);
    else if (bin->size*FILL_MIN_DENOM < bin->cap*FILL_MIN_NOM)
	resize_lck(bin, bin->cap/2);
}

cuflow_cacheobj_t
cuflow_cache_call(cuflow_cache_t cache, cuflow_cacheobj_t key)
{
    cu_hash_t hash = cacheobj_hash(key);
    cuflowP_cachebin_t bin;
    cuflow_cacheobj_t *slot;
    cuflow_cacheobj_t obj;
    cu_word_t fncode;

    /* Lookup key, return if found. */
    bin = &cache->bin_arr[HASH_BIN(hash)];
    cu_mutex_lock(&bin->mutex);
    slot = &bin->link_arr[HASH_SLOT(hash, bin->cap)];
    obj = *slot;
    while (obj) {
	if (cacheobj_eq(obj, key)) {
	    float t = (float)cuflow_walltime();
	    update(cache->conf, obj, t);
	    HDR(obj)->access_function += 1.0;
	    cu_mutex_unlock(&bin->mutex);
	    return obj;
	}
	obj = HDR(obj)->next;
    }

    /* Not found, prune and adjust if needed, cache computation. */
    fncode = CUFLOW_CACHEOBJ_FNCODE(obj);
    if (bin->size*FILL_MAX_DENOM > bin->cap*FILL_MAX_NOM)
	prune_and_adjust_lck(cache, bin);
    obj = cu_call(cache->fn_arr[CUFLOW_FNCODE_SLOT(fncode)], key);

    /* Insert new object. */
    HDR(obj)->next = *slot;
    if (HDR(obj)->gain == 0)
	cu_bugf("Callback %d for cache object did not set gain.",
		CUFLOW_FNCODE_SLOT(fncode));
    HDR(obj)->t_access = cuflow_walltime();
    HDR(obj)->access_function = 1.0;
    *slot = obj;
    ++bin->size;
    cu_mutex_unlock(&bin->mutex);
    return obj;
}
